/* ui-icons.c
 * This file is part of Vara digital painting program
 * Copyright 2021, 2023, 2024, 2025 Nandakumar Edamana
 * Distributed under GNU General Public License, version 3
 */ 

#include <assert.h>
#include <stdarg.h>
#include <stdbool.h>

#include "ui-icons.h"


void tool_icons_destruct(ToolIcons *this)
{
	if(this->img_pencil) {
		image_destruct((Image *) this->img_pencil);
		free(this->img_pencil);
	}

	if(this->img_brush) {
		image_destruct((Image *) this->img_brush);
		free(this->img_brush);
	}

	if(this->img_eraser) {
		image_destruct((Image *) this->img_eraser);
		free(this->img_eraser);
	}

	if(this->img_eyedropper) {
		image_destruct((Image *) this->img_eyedropper);
		free(this->img_eyedropper);
	}

	if(this->img_floodfill) {
		image_destruct((Image *) this->img_floodfill);
		free(this->img_floodfill);
	}

	if(this->img_pan) {
		image_destruct((Image *) this->img_pan);
		free(this->img_pan);
	}
}

void tool_icons_construct(ToolIcons *this)
{
	this->img_pan = NULL;
	this->img_floodfill = NULL;
	this->img_eyedropper = NULL;
	this->img_eraser = NULL;
	this->img_brush = NULL;
	this->img_pencil = NULL;
}

void helper_draw_line_perc(ImageDocument *doc, double x1, double y1, double p1, double x2, double y2, double p2)
{
	doc->drawparams.oldpressure = p1;

	doc->drawparams.pressure = p2;

	VaraRect rect = image_line(image_document_get_curlayer(doc)->img, roundedpercsiz(x1, image_document_get_width(doc)), roundedpercsiz(y1, image_document_get_height(doc)), roundedpercsiz(x2, image_document_get_width(doc)), roundedpercsiz(y2, image_document_get_height(doc)), doc->drawparams);

	image_document_add_dirty_region(doc, rect);
}

void helper_set_fill_size_perc(ImageDocument *doc, double siz)
{
	doc->drawparams.brushparams.siz = roundedpercsiz(siz, image_document_get_width(doc));
}

void helper_set_stroke_color(ImageDocument *doc, float r, float g, float b)
{
	doc->drawparams.fgcolor = srgbpixel2linear((Pixel) { {r, g, b, 1} });
}

ToolIcons * draw_icons()
{
	ToolIcons *toolicons = (ToolIcons *) malloc(sizeof(ToolIcons));
	if(toolicons == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	tool_icons_construct(toolicons);

	int iconsiz = 24;
	double padding = 0.05;
	double stroke_width = 0.12;
	double fill_size = 0.3;

	ImageDocument *doc = (ImageDocument *) malloc(sizeof(ImageDocument));
	if(doc == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	image_document_construct(doc, iconsiz, iconsiz, default_drawparams(), true);

	doc->drawparams.active_tool = TOOL_BRUSH;

	doc->drawparams.brushparams = prebrush_round();
	doc->drawparams.brushparams.pressure_size = true;

	double pencilnecky = 0.8;
	double pencil_fill_size = fill_size * 0.8;

	image_document_clear_layer(doc);

	helper_set_fill_size_perc(doc, pencil_fill_size + stroke_width);
	helper_set_stroke_color(doc, 0, 0, 0);
	helper_draw_line_perc(doc, 0.5, pencilnecky, 1, 0.5, padding, 0);
	helper_draw_line_perc(doc, 0.5, pencilnecky, 1, 0.5, 1, 1);

	helper_set_fill_size_perc(doc, pencil_fill_size);
	helper_set_stroke_color(doc, 0.5, 0.5, 0.5);
	helper_draw_line_perc(doc, 0.5, pencilnecky, 1, 0.5, 1, 1);
	helper_set_stroke_color(doc, 0.7, 0.7, 0.7);
	helper_draw_line_perc(doc, 0.5, pencilnecky, 1, 0.5, padding + stroke_width, 0);

	helper_set_fill_size_perc(doc, 0.15);
	helper_set_stroke_color(doc, 0, 0, 0);
	helper_draw_line_perc(doc, 0.5, padding + stroke_width + 0.3, 1, 0.5, padding + stroke_width, 0);

	Image *_ngg_tmp_1 = image_make_clone(image_document_get_fglayer_must(doc)->img);
	if(toolicons->img_pencil) {
		image_destruct((Image *) toolicons->img_pencil);
		free(toolicons->img_pencil);
	}

	toolicons->img_pencil = _ngg_tmp_1;

	double brush_gap = stroke_width * 1.2;
	double handle_size = fill_size * 0.5;

	image_document_clear_layer(doc);

	helper_set_stroke_color(doc, 0, 0, 0);
	helper_set_fill_size_perc(doc, handle_size + stroke_width);
	helper_draw_line_perc(doc, 0.5, pencilnecky + brush_gap, 1, 0.5, 1, 1);

	helper_set_stroke_color(doc, 0, 0, 0);
	helper_set_fill_size_perc(doc, fill_size + stroke_width);
	helper_draw_line_perc(doc, 0.5, pencilnecky - brush_gap, 1, 0.5, padding, 0);
	helper_set_stroke_color(doc, 0.5, 0.5, 0.5);
	helper_set_fill_size_perc(doc, handle_size);
	helper_draw_line_perc(doc, 0.5, pencilnecky + brush_gap, 1, 0.5, 1, 1);

	helper_set_stroke_color(doc, 0, 0.5, 1);
	helper_set_fill_size_perc(doc, fill_size);
	helper_draw_line_perc(doc, 0.5, pencilnecky - brush_gap, 1, 0.5, padding + stroke_width, 0);

	Image *_ngg_tmp_2 = image_make_clone(image_document_get_fglayer_must(doc)->img);
	if(toolicons->img_brush) {
		image_destruct((Image *) toolicons->img_brush);
		free(toolicons->img_brush);
	}

	toolicons->img_brush = _ngg_tmp_2;

	double eraser_fill_size = fill_size * 1.5;
	double erasertopy = (fill_size / 2.0) + padding + stroke_width;

	image_document_clear_layer(doc);

	helper_set_fill_size_perc(doc, eraser_fill_size);
	helper_set_stroke_color(doc, 0, 0, 0);
	helper_draw_line_perc(doc, 0.5, erasertopy, 1, 0.5, 1 - erasertopy, 1);

	helper_set_fill_size_perc(doc, eraser_fill_size - stroke_width);
	helper_set_stroke_color(doc, 0.5, 0.5, 0.5);
	helper_draw_line_perc(doc, 0.5, erasertopy, 1, 0.5, 0.5, 1);
	helper_set_stroke_color(doc, 1, 1, 1);
	helper_draw_line_perc(doc, 0.5, 0.5, 1, 0.5, 1 - erasertopy, 1);

	Image *_ngg_tmp_3 = image_make_clone(image_document_get_fglayer_must(doc)->img);
	if(toolicons->img_eraser) {
		image_destruct((Image *) toolicons->img_eraser);
		free(toolicons->img_eraser);
	}

	toolicons->img_eraser = _ngg_tmp_3;

	image_document_clear_layer(doc);

	double eyedropper_fill_size = pencil_fill_size;
	double eyedroppernecky = pencilnecky - 0.5;
	double eyedroppertipy = pencilnecky - 0.2;
	double eyedropperdroptop = eyedroppertipy + 0.01;
	double eyedropper_fill_radius = eyedropper_fill_size * 0.5;
	double eyedropperdropend = 1 - eyedropper_fill_radius - stroke_width;

	helper_set_fill_size_perc(doc, eyedropper_fill_size + stroke_width);
	helper_set_stroke_color(doc, 0, 0, 0);
	helper_draw_line_perc(doc, 0.5, eyedroppernecky, 1, 0.5, 0, 1);
	helper_draw_line_perc(doc, 0.5, eyedroppernecky, 1, 0.5, eyedroppertipy + stroke_width, 0);
	helper_draw_line_perc(doc, 0.5, eyedropperdroptop, 0, 0.5, eyedropperdropend, 1);
	helper_set_fill_size_perc(doc, pencil_fill_size);
	helper_set_stroke_color(doc, 0.5, 0.5, 0.5);
	helper_draw_line_perc(doc, 0.5, eyedroppernecky, 1, 0.5, 0, 1);
	helper_draw_line_perc(doc, 0.5, eyedroppernecky, 1, 0.5, eyedroppertipy, 0);
	helper_set_stroke_color(doc, 0, 0.5, 1);
	helper_draw_line_perc(doc, 0.5, eyedropperdroptop, 0, 0.5, eyedropperdropend, 1);

	Image *_ngg_tmp_4 = image_make_clone(image_document_get_fglayer_must(doc)->img);
	if(toolicons->img_eyedropper) {
		image_destruct((Image *) toolicons->img_eyedropper);
		free(toolicons->img_eyedropper);
	}

	toolicons->img_eyedropper = _ngg_tmp_4;

	image_document_clear_layer(doc);

	double bucket_fill_size = 0.6;

	helper_set_fill_size_perc(doc, bucket_fill_size);
	helper_set_stroke_color(doc, 0, 0, 0);
	helper_draw_line_perc(doc, 0.4, erasertopy, 1, 0.6, 1 - erasertopy, 1);

	helper_set_fill_size_perc(doc, bucket_fill_size - stroke_width);
	helper_set_stroke_color(doc, 0.5, 0.5, 0.5);
	helper_draw_line_perc(doc, 0.4, erasertopy, 1, 0.6, 1 - erasertopy, 1);

	helper_set_fill_size_perc(doc, bucket_fill_size);
	helper_set_stroke_color(doc, 0, 0, 0);
	helper_draw_line_perc(doc, 0.4, erasertopy, 1, 0.4, erasertopy, 1);

	helper_set_fill_size_perc(doc, bucket_fill_size - stroke_width);
	helper_set_stroke_color(doc, 0, 0.5, 1);
	helper_draw_line_perc(doc, 0.4, erasertopy, 1, 0.4, erasertopy, 1);

	helper_set_stroke_color(doc, 0, 0, 0);
	helper_set_fill_size_perc(doc, eyedropper_fill_size + stroke_width);
	helper_draw_line_perc(doc, 0.3, 0.48, 0, 0.3, 0.8, 1);

	helper_set_stroke_color(doc, 0, 0.5, 1);
	helper_set_fill_size_perc(doc, pencil_fill_size);
	helper_draw_line_perc(doc, 0.3, 0.45, 0, 0.3, 0.8, 1);

	Image *_ngg_tmp_5 = image_make_clone(image_document_get_fglayer_must(doc)->img);
	if(toolicons->img_floodfill) {
		image_destruct((Image *) toolicons->img_floodfill);
		free(toolicons->img_floodfill);
	}

	toolicons->img_floodfill = _ngg_tmp_5;

	image_document_clear_layer(doc);

	helper_set_stroke_color(doc, 0, 0, 0);
	helper_set_fill_size_perc(doc, pencil_fill_size);
	helper_draw_line_perc(doc, 0.5, 0.2, 1, 0.5, 0.8, 1);
	helper_draw_line_perc(doc, 0.2, 0.5, 1, 0.8, 0.5, 1);
	helper_set_stroke_color(doc, 0.5, 0.5, 0.5);
	helper_set_fill_size_perc(doc, pencil_fill_size - stroke_width);
	helper_draw_line_perc(doc, 0.5, 0.2, 1, 0.5, 0.8, 1);
	helper_draw_line_perc(doc, 0.2, 0.5, 1, 0.8, 0.5, 1);

	Image *_ngg_tmp_6 = image_make_clone(image_document_get_fglayer_must(doc)->img);
	if(toolicons->img_pan) {
		image_destruct((Image *) toolicons->img_pan);
		free(toolicons->img_pan);
	}

	toolicons->img_pan = _ngg_tmp_6;

	if(doc) {
		image_document_destruct(doc);
		free(doc);
	}

	return toolicons;
}
