/* xcf-read.c
 * This file is part of Vara digital painting program
 * Copyright 2021, 2023, 2024, 2025 Nandakumar Edamana
 * Distributed under GNU General Public License, version 3
 */ 

#include <assert.h>
#include <stdarg.h>
#include <stdbool.h>

#include "xcf-read.h"


#include <arpa/inet.h>

#include <glib.h>

_Bool xcf_log_enabled;

void xcf_layer_set_alpha(XcfLayer *this, double alpha_in)
{
	this->alpha = alpha_in;
}

void xcf_layer_set_visible(XcfLayer *this, _Bool visible_in)
{
	this->visible = visible_in;
}

void xcf_layer_construct(XcfLayer *this, char * name, int width, int height, XcfTileVector *tiles)
{
	this->visible = true;
	this->alpha = 1.0;
	this->name = name;
	this->width = width;
	this->height = height;
	this->tiles = tiles;
}

void xcf_layer_destruct(XcfLayer *this)
{
	if(this->name) {
		free(this->name);
	}

	if(this->tiles) {
		xcf_tile_vector_destruct(this->tiles);
		free(this->tiles);
	}
}

void xcf_layer_vector_destruct(XcfLayerVector *this)
{
	xcf_layer_vector_delete_all(this);
	if(this->arr) {
		free(this->arr);
	}
}

void xcf_layer_vector__resize(XcfLayerVector *this, int newcount)
{
	if(newcount > this->alloccount) {
		int alloccountbak = this->alloccount;

		this->alloccount = 2 * this->alloccount;

		assert(this->alloccount > alloccountbak); /* owningvector.ngg:25 */

		assert(newcount <= this->alloccount); /* owningvector.ngg:26 */

		this->arr = realloc(this->arr, this->alloccount * sizeof(this->arr[0]));
		if(this->arr == NULL) { perror(NULL); exit(EXIT_FAILURE); }

	}

	this->count = newcount;
}

void xcf_layer_vector_append(XcfLayerVector *this, XcfLayer *newitem)
{
	int newcount = 1 + this->count;

	xcf_layer_vector__resize(this, newcount);

	this->arr[newcount - 1] = newitem;
}

void xcf_layer_vector_clear(XcfLayerVector *this)
{
	xcf_layer_vector_delete_all(this);

	this->count = 0;
	this->arr = realloc(this->arr, 8 * sizeof(this->arr[0]));
	if(this->arr == NULL) { perror(NULL); exit(EXIT_FAILURE); }

	this->alloccount = 8;
}

void xcf_layer_vector_delete_all(XcfLayerVector *this)
{
	while(this->count > 0) {
		XcfLayer *obj = xcf_layer_vector_pop(this);
		if(obj) {
			xcf_layer_destruct(obj);
			free(obj);
		}
	}
}

XcfLayer * xcf_layer_vector_get_item(XcfLayerVector *this, int index)
{
	assert(index < this->count); /* owningvector.ngg:58 */

	return this->arr[index];
}

XcfLayer * xcf_layer_vector_pop(XcfLayerVector *this)
{
	assert(this->count > 0); /* owningvector.ngg:64 */
	XcfLayer *r = xcf_layer_vector_get_item(this, this->count - 1);
	this->count = this->count - 1;

	return r;
}

void xcf_layer_vector_set_item(XcfLayerVector *this, int index, XcfLayer *itm)
{
	assert(index < this->count); /* owningvector.ngg:77 */

	XcfLayer *todel = this->arr[index];

	this->arr[index] = itm;
	if(todel) {
		xcf_layer_destruct(todel);
		free(todel);
	}
}

int xcf_layer_vector_get_count(XcfLayerVector *this)
{
	return this->count;
}

_Bool xcf_layer_vector_is_empty(XcfLayerVector *this)
{
	return 0 == this->count;
}

void xcf_layer_vector_construct(XcfLayerVector *this)
{
	this->alloccount = 8;
	XcfLayer **_tmp_1 = (XcfLayer * *) calloc((size_t) 8, sizeof(XcfLayer *));
	if(_tmp_1 == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	this->arr = _tmp_1;
	this->count = 0;
}

void xcf_property_construct(XcfProperty *this, unsigned int id, unsigned int length, XcfPropertyValue value)
{
	this->next = NULL;
	this->id = id;
	this->length = length;
	this->value = value;
}

XcfProperty * xcf_property_get_next(XcfProperty *this)
{
	return this->next;
}

void xcf_property_destruct(XcfProperty *this)
{
	if(this->next) {
		xcf_property_destruct((XcfProperty *) this->next);
		free(this->next);
	}
}

void xcf_pointer_vector__resize(XcfPointerVector *this, int newcount)
{
	if(newcount > this->alloccount) {
		int alloccountbak = this->alloccount;

		this->alloccount = 2 * this->alloccount;

		assert(this->alloccount > alloccountbak); /* vector.ngg:17 */

		assert(newcount <= this->alloccount); /* vector.ngg:18 */

		this->arr = realloc(this->arr, this->alloccount * sizeof(this->arr[0]));
		if(this->arr == NULL) { perror(NULL); exit(EXIT_FAILURE); }

	}

	this->count = newcount;
}

void xcf_pointer_vector_append(XcfPointerVector *this, long long newitem)
{
	int newcount = 1 + this->count;

	xcf_pointer_vector__resize(this, newcount);

	this->arr[newcount - 1] = newitem;
}

void xcf_pointer_vector_clear(XcfPointerVector *this)
{
	this->count = 0;
	this->alloccount = 8;
	this->arr = realloc(this->arr, this->alloccount * sizeof(this->arr[0]));
	if(this->arr == NULL) { perror(NULL); exit(EXIT_FAILURE); }

}

long long xcf_pointer_vector_get_item(XcfPointerVector *this, int index)
{
	assert(index < this->count); /* vector.ngg:43 */

	return this->arr[index];
}

long long xcf_pointer_vector_pop(XcfPointerVector *this)
{
	assert(this->count > 0); /* vector.ngg:49 */
	long long r = xcf_pointer_vector_get_item(this, this->count - 1);
	this->count = this->count - 1;

	return r;
}

void xcf_pointer_vector_set_item(XcfPointerVector *this, int index, long long itm)
{
	assert(index < this->count); /* vector.ngg:62 */

	this->arr[index] = itm;
}

int xcf_pointer_vector_get_count(XcfPointerVector *this)
{
	return this->count;
}

_Bool xcf_pointer_vector_is_empty(XcfPointerVector *this)
{
	return 0 == this->count;
}

void xcf_pointer_vector_destruct(XcfPointerVector *this)
{
	if(this->arr) {
		free(this->arr);
	}
}

void xcf_pointer_vector_construct(XcfPointerVector *this)
{
	this->alloccount = 8;
	long long *_tmp_1 = (long long *) calloc((size_t) 8, sizeof(long long));
	if(_tmp_1 == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	this->arr = _tmp_1;
	this->count = 0;
}

void xcf_tile_construct(XcfTile *this, int width, int height)
{
	assert(width >= 1 && width <= 64); /* xcf-read.ngg:77 */

	assert(height >= 1 && height <= 64); /* xcf-read.ngg:78 */

	this->width = width;

	this->height = height;

	Image *_tmp_1 = (Image *) malloc(sizeof(Image));
	if(_tmp_1 == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	image_construct(_tmp_1, width, height, 255);
	this->img = _tmp_1;
}

float xcf_tile_get_pixel_component(XcfTile *this, int x, int y, int channel)
{
	return this->buf[(y * this->width) + x].rgb[channel];
}

void xcf_tile_set_byte(XcfTile *this, int x, int y, int channel, int bytidx, unsigned char bytval)
{
	float comp = this->buf[(y * this->width) + x].rgb[channel];
	comp = set_byte_in_float(comp, bytidx, bytval);
	this->buf[(y * this->width) + x].rgb[channel] = comp;
}

void xcf_tile_scale_pixel_alpha_component_from_byte(XcfTile *this, int x, int y)
{
	this->buf[(y * this->width) + x].rgb[3] = get_byte_in_float(this->buf[(y * this->width) + x].rgb[3], 0) / 255.0;
}

void xcf_tile_scale_pixel_color_components_from_gamma_byte(XcfTile *this, int x, int y)
{
	int channel;
	for(channel = 0; channel < 3; channel += 1) {
		this->buf[(y * this->width) + x].rgb[channel] = srgb2linear(get_byte_in_float(this->buf[(y * this->width) + x].rgb[channel], 0) / 255.0);
	}
}

void xcf_tile_scale_pixel_color_components_from_linear_byte(XcfTile *this, int x, int y)
{
	int channel;
	for(channel = 0; channel < 3; channel += 1) {
		this->buf[(y * this->width) + x].rgb[channel] = get_byte_in_float(this->buf[(y * this->width) + x].rgb[channel], 0) / 255.0;
	}
}

void xcf_tile_set_pixel_component(XcfTile *this, int x, int y, int channel, float value)
{
	this->buf[(y * this->width) + x].rgb[channel] = value;
}

void xcf_tile_finish(XcfTile *this)
{
	int y;
	for(y = 0; y < this->height; y += 1) {
		int x;
		for(x = 0; x < this->width; x += 1) {
			int c;
			LinearPixel px = this->buf[(y * this->width) + x];

			for(c = 0; c < 3; c += 1) {
				px.rgb[c] = px.rgb[c] * px.rgb[3];
			}

			nan_img_set_linear_pixel(this->img, x, y, px);
		}
	}
}

void xcf_tile_destruct(XcfTile *this)
{
	if(this->img) {
		image_destruct(this->img);
		free(this->img);
	}
}

void xcf_tile_vector_destruct(XcfTileVector *this)
{
	xcf_tile_vector_delete_all(this);
	if(this->arr) {
		free(this->arr);
	}
}

void xcf_tile_vector__resize(XcfTileVector *this, int newcount)
{
	if(newcount > this->alloccount) {
		int alloccountbak = this->alloccount;

		this->alloccount = 2 * this->alloccount;

		assert(this->alloccount > alloccountbak); /* owningvector.ngg:25 */

		assert(newcount <= this->alloccount); /* owningvector.ngg:26 */

		this->arr = realloc(this->arr, this->alloccount * sizeof(this->arr[0]));
		if(this->arr == NULL) { perror(NULL); exit(EXIT_FAILURE); }

	}

	this->count = newcount;
}

void xcf_tile_vector_append(XcfTileVector *this, XcfTile *newitem)
{
	int newcount = 1 + this->count;

	xcf_tile_vector__resize(this, newcount);

	this->arr[newcount - 1] = newitem;
}

void xcf_tile_vector_clear(XcfTileVector *this)
{
	xcf_tile_vector_delete_all(this);

	this->count = 0;
	this->arr = realloc(this->arr, 8 * sizeof(this->arr[0]));
	if(this->arr == NULL) { perror(NULL); exit(EXIT_FAILURE); }

	this->alloccount = 8;
}

void xcf_tile_vector_delete_all(XcfTileVector *this)
{
	while(this->count > 0) {
		XcfTile *obj = xcf_tile_vector_pop(this);
		if(obj) {
			xcf_tile_destruct(obj);
			free(obj);
		}
	}
}

XcfTile * xcf_tile_vector_get_item(XcfTileVector *this, int index)
{
	assert(index < this->count); /* owningvector.ngg:58 */

	return this->arr[index];
}

XcfTile * xcf_tile_vector_pop(XcfTileVector *this)
{
	assert(this->count > 0); /* owningvector.ngg:64 */
	XcfTile *r = xcf_tile_vector_get_item(this, this->count - 1);
	this->count = this->count - 1;

	return r;
}

void xcf_tile_vector_set_item(XcfTileVector *this, int index, XcfTile *itm)
{
	assert(index < this->count); /* owningvector.ngg:77 */

	XcfTile *todel = this->arr[index];

	this->arr[index] = itm;
	if(todel) {
		xcf_tile_destruct(todel);
		free(todel);
	}
}

int xcf_tile_vector_get_count(XcfTileVector *this)
{
	return this->count;
}

_Bool xcf_tile_vector_is_empty(XcfTileVector *this)
{
	return 0 == this->count;
}

void xcf_tile_vector_construct(XcfTileVector *this)
{
	this->alloccount = 8;
	XcfTile **_tmp_1 = (XcfTile * *) calloc((size_t) 8, sizeof(XcfTile *));
	if(_tmp_1 == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	this->arr = _tmp_1;
	this->count = 0;
}

void raw_xcf_document_destruct(RawXcfDocument *this)
{
	if(this->layers) {
		xcf_layer_vector_destruct(this->layers);
		free(this->layers);
	}
}

void raw_xcf_document_construct(RawXcfDocument *this)
{
	XcfLayerVector *_tmp_1 = (XcfLayerVector *) malloc(sizeof(XcfLayerVector));
	if(_tmp_1 == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	xcf_layer_vector_construct(_tmp_1);
	this->layers = _tmp_1;
	this->hdr = xcf_header_default();
}

long long get_max_pointer(FILE * fp)
{
	long bakpos = ftell(fp);

	fseek(fp, 0, SEEK_END);
	long endptr = ftell(fp);

	fseek(fp, bakpos, SEEK_SET);

	return endptr;
}

ImageDocument * image_document_from_raw_xcf_document(RawXcfDocument *rawdoc, const char * path)
{
	int i;
	ImageDocument *imgdoc = (ImageDocument *) malloc(sizeof(ImageDocument));
	if(imgdoc == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	image_document_construct(imgdoc, (int) ntohl(rawdoc->hdr.width), (int) ntohl(rawdoc->hdr.height), default_drawparams(), false);

	for(i = xcf_layer_vector_get_count(rawdoc->layers) - 1; i >= 0; i -= 1) {
		int tiley;
		XcfLayer *inlayer = xcf_layer_vector_get_item(rawdoc->layers, i);
		Layer *outlayer = (Layer *) malloc(sizeof(Layer));
		if(outlayer == NULL) {
			perror(NULL);
			exit(EXIT_FAILURE);
		}

		layer_construct(outlayer, inlayer->name, inlayer->width, inlayer->height, LINEAR_PIXEL_TRANSP);

		layer_set_alpha(outlayer, inlayer->alpha);
		layer_set_visible(outlayer, inlayer->visible);

		int ntileh = (int) ceil(inlayer->width / 64.0);
		int ntilev = (int) ceil(inlayer->height / 64.0);

		int yoffset = 0;
		int xoffset = 0;

		for(tiley = 0; tiley < ntilev; tiley += 1) {
			int tilex;
			xoffset = 0;

			for(tilex = 0; tilex < ntileh; tilex += 1) {
				XcfTile *tile = xcf_tile_vector_get_item(inlayer->tiles, (tiley * ntileh) + tilex);
				image_copy_image_region_with_destoffset(outlayer->img, tile->img, (VaraRect) { 0, 0, imin2(64, tile->width), imin2(64, tile->height) }, xoffset, yoffset);

				xoffset += 64;
			}

			yoffset += 64;
		}

		image_document_append_layer(imgdoc, outlayer);
	}

	if(layer_owning_vector_get_count(imgdoc->layers) < 1) {
		perrorxcf(path, NULL, "no layers read");
		ImageDocument *_tmp_1;
		_tmp_1 = imgdoc;
		imgdoc = NULL;
		ImageDocument *todel = _tmp_1;
		if(todel) {
			image_document_destruct(todel);
			free(todel);
		}
	}

	ImageDocument *_tmp_1;
	_tmp_1 = imgdoc;
	imgdoc = NULL;
	return _tmp_1;
}

void logxcf(const char * path, const char * fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	if(xcf_log_enabled) {
		size_t _tmp_1 = strlen(path);
		size_t _tmp_2 = strlen(fmt);
		size_t _tmp_3 = strlen("\n");
		size_t _tmp_catpos = 0u;
		char *_tmp_4 = (char *) calloc((size_t) (_tmp_3 + _tmp_2 + (2u) + _tmp_1 + (13u) + (1u)), sizeof(char));
		if(_tmp_4 == NULL) {
			perror(NULL);
			exit(EXIT_FAILURE);
		}

		char * _tmp_catstr = (char *) _tmp_4;
		memcpy(_tmp_catstr + _tmp_catpos, "xcf reading: ", 13u);
		_tmp_catpos += (13u);
		memcpy(_tmp_catstr + _tmp_catpos, path, _tmp_1);
		_tmp_catpos += _tmp_1;
		memcpy(_tmp_catstr + _tmp_catpos, ": ", 2u);
		_tmp_catpos += (2u);
		memcpy(_tmp_catstr + _tmp_catpos, fmt, _tmp_2);
		_tmp_catpos += _tmp_2;
		memcpy(_tmp_catstr + _tmp_catpos, "\n", _tmp_3);
		_tmp_catpos += _tmp_3;
		_tmp_catstr[_tmp_catpos] = '\0';
		char * _ngg_tmp_7 = _tmp_catstr;
		vfprintf(stderr, _ngg_tmp_7, ap);
		if(_ngg_tmp_7) {
			free(_ngg_tmp_7);
		}
	}
}

void perrorxcf(const char * path, FILE * fp, const char * fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	if(fp) {
		FILE * fpnn;
		fpnn = (FILE *) fp;
		fprintf(stderr, "at offset %ld: ", ftell(fpnn));
	}

	size_t _tmp_1 = strlen(fmt);
	size_t _tmp_2 = strlen("\n");
	size_t _tmp_catpos = 0u;
	char *_tmp_3 = (char *) calloc((size_t) (_tmp_2 + _tmp_1 + (19u) + (1u)), sizeof(char));
	if(_tmp_3 == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	char * _tmp_catstr = (char *) _tmp_3;
	memcpy(_tmp_catstr + _tmp_catpos, "error reading xcf: ", 19u);
	_tmp_catpos += (19u);
	memcpy(_tmp_catstr + _tmp_catpos, fmt, _tmp_1);
	_tmp_catpos += _tmp_1;
	memcpy(_tmp_catstr + _tmp_catpos, "\n", _tmp_2);
	_tmp_catpos += _tmp_2;
	_tmp_catstr[_tmp_catpos] = '\0';
	char * _ngg_tmp_8 = _tmp_catstr;
	vfprintf(stderr, _ngg_tmp_8, ap);
	if(_ngg_tmp_8) {
		free(_ngg_tmp_8);
	}
}

_Bool read_and_ignore_xcf_pointers_from_fp(FILE * fp, const char * purpose, const char * path)
{
	XcfPointerVector *vecptrs = read_xcf_pointers_from_fp(fp, path, purpose);

	if(vecptrs) {
		XcfPointerVector *vecptrsnn;
		vecptrsnn = (XcfPointerVector *) vecptrs;
		if(xcf_pointer_vector_get_count(vecptrsnn) == 0) {
			if(vecptrs) {
				xcf_pointer_vector_destruct((XcfPointerVector *) vecptrs);
				free(vecptrs);
			}

			return true;
		}

		logxcf(path, "%s are not supported", purpose);
		if(vecptrs) {
			xcf_pointer_vector_destruct((XcfPointerVector *) vecptrs);
			free(vecptrs);
		}

		return true;
	}

	if(vecptrs) {
		xcf_pointer_vector_destruct((XcfPointerVector *) vecptrs);
		free(vecptrs);
	}

	return false;
}

_ngg_tuple_read_dimentions_from_xcf_fp read_dimentions_from_xcf_fp(FILE * fp, const char * path)
{
	unsigned int widthbe = 0u;
	unsigned int heightbe = widthbe;

	fread(&widthbe, sizeof(widthbe), 1, fp);
	if(feof(fp) || ferror(fp)) {
		perrorxcf(path, fp, "could not read width");
		return (_ngg_tuple_read_dimentions_from_xcf_fp){0, 0, false};
	}

	fread(&heightbe, sizeof(heightbe), 1, fp);
	if(feof(fp) || ferror(fp)) {
		perrorxcf(path, fp, "could not read height");
		return (_ngg_tuple_read_dimentions_from_xcf_fp){0, 0, false};
	}

	return (_ngg_tuple_read_dimentions_from_xcf_fp){(int) ntohl(widthbe), (int) ntohl(heightbe), true};
}

ImageDocument * read_image_document_from_xcf(const char * path)
{
	FILE * fpnul = fopen(path, "rb");
	if(fpnul) {
		FILE * fp;
		fp = (FILE *) fpnul;
		ImageDocument *retval = read_image_document_from_xcf_fp(fp, path);
		fclose(fp);
		return retval;
	}

	perrorxcf(path, NULL, "fopen() failed; message: %s", strerror(errno));

	return NULL;
}

ImageDocument * read_image_document_from_xcf_fp(FILE * fp, const char * path)
{
	RawXcfDocument *rawdocnul = read_raw_xcf_document_from_fp(fp, path);

	if(rawdocnul) {
		ImageDocument *tmp = image_document_from_raw_xcf_document(rawdocnul, path);
		if(rawdocnul) {
			raw_xcf_document_destruct((RawXcfDocument *) rawdocnul);
			free(rawdocnul);
		}

		return tmp;
	}

	if(rawdocnul) {
		raw_xcf_document_destruct((RawXcfDocument *) rawdocnul);
		free(rawdocnul);
	}

	return NULL;
}

RawXcfDocument * read_raw_xcf_document_from_fp(FILE * fp, const char * path)
{
	int i;
	long long lptr;
	XcfHeader hdr = xcf_header_default();

	fread(&hdr, sizeof(hdr), 1, fp);
	if(feof(fp) || ferror(fp)) {
		perrorxcf(path, fp, "could not read the header");
		return NULL;
	}

	if(0 != memcmp((const char *) hdr.magic, "gimp xcf ", 9u)) {
		perrorxcf(path, fp, "invalid header");
		return NULL;
	}

	if(0 != memcmp((const char *) hdr.version, "v011\0", 5u)) {
		perrorxcf(path, fp, "unsupported version");
		return NULL;
	}

	unsigned int precisionhbo = ntohl(hdr.precision);
	_ngg_tuple_decode_xcf_precision _ngg_tmp_0 = decode_xcf_precision(precisionhbo);
	_Bool precok = _ngg_tmp_0.m1;
	XcfPrecision precision = _ngg_tmp_0.m0;
	if(!precok) {
		perrorxcf(path, fp, "unsupported precision: %u", precisionhbo);
		return NULL;
	}

	logxcf(path, "header read: %ux%u; base_type: %u; precision: %u", ntohl(hdr.width), ntohl(hdr.height), ntohl(hdr.base_type), precisionhbo);

	int vara_nbit = (int) ((8u) * sizeof(float));
	if(precision.bits != vara_nbit) {
		logxcf(path, "bit depth mismatch; upscaling/downscaling from %u to %u will happen", precision.bits, vara_nbit);
	}

	if(!precision.linear) {
		logxcf(path, "conversion from gamma to linear will happen");
	}

	XcfProperty *propshead = read_raw_xcf_properties_from_fp(fp, path);
	if(!propshead) {
		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		return NULL;
	}

	XcfPointerVector *layerptrs = read_xcf_pointers_from_fp(fp, path, "layer");
	if(!layerptrs) {
		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		if(layerptrs) {
			xcf_pointer_vector_destruct((XcfPointerVector *) layerptrs);
			free(layerptrs);
		}

		return NULL;
	}

	if(!read_and_ignore_xcf_pointers_from_fp(fp, "channels", path)) {
		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		if(layerptrs) {
			xcf_pointer_vector_destruct((XcfPointerVector *) layerptrs);
			free(layerptrs);
		}

		return NULL;
	}

	assert(layerptrs);
	XcfPointerVector *layerptrsnn = layerptrs;

	logxcf(path, "layer count: %d", xcf_pointer_vector_get_count(layerptrsnn));

	RawXcfDocument *rawdoc = (RawXcfDocument *) malloc(sizeof(RawXcfDocument));
	if(rawdoc == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	raw_xcf_document_construct(rawdoc);

	rawdoc->hdr = hdr;

	for(i = 0; i < xcf_pointer_vector_get_count(layerptrsnn); i += 1) {
		lptr = xcf_pointer_vector_get_item(layerptrsnn, i);
		logxcf(path, "reading layer %d (offset: %lld)", i, lptr);

		XcfLayer *layer = read_xcf_layer_from_fp(fp, lptr, precision, path);
		if(layer) {
			XcfLayer *layernn;
			layernn = (XcfLayer *) layer;
			xcf_layer_vector_append(rawdoc->layers, layernn);
		} else {
			if(propshead) {
				xcf_property_destruct((XcfProperty *) propshead);
				free(propshead);
			}

			if(layerptrs) {
				xcf_pointer_vector_destruct((XcfPointerVector *) layerptrs);
				free(layerptrs);
			}

			if(rawdoc) {
				raw_xcf_document_destruct(rawdoc);
				free(rawdoc);
			}

			return NULL;
		}
	}

	if(xcf_layer_vector_get_count(rawdoc->layers) < 1) {
		perrorxcf(path, fp, "has zero layers");
		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		if(layerptrs) {
			xcf_pointer_vector_destruct((XcfPointerVector *) layerptrs);
			free(layerptrs);
		}

		if(rawdoc) {
			raw_xcf_document_destruct(rawdoc);
			free(rawdoc);
		}

		return NULL;
	}

	RawXcfDocument *_tmp_1;
	_tmp_1 = rawdoc;
	rawdoc = NULL;
	if(propshead) {
		xcf_property_destruct((XcfProperty *) propshead);
		free(propshead);
	}

	if(layerptrs) {
		xcf_pointer_vector_destruct((XcfPointerVector *) layerptrs);
		free(layerptrs);
	}

	if(rawdoc) {
		raw_xcf_document_destruct(rawdoc);
		free(rawdoc);
	}

	return _tmp_1;
}

char * read_string_pointers_from_xcf_fp(FILE * fp, const char * path)
{
	unsigned int lenbe = 0u;

	fread(&lenbe, sizeof(lenbe), 1, fp);
	if(feof(fp) || ferror(fp)) {
		perrorxcf(path, fp, "could not read string size");
		return NULL;
	}

	unsigned int lenhbo = ntohl(lenbe);
	if(lenhbo > MAX_LAYERNAM) {
		perrorxcf(path, fp, "layer name with length > %u", MAX_LAYERNAM);
		return NULL;
	}

	char *buf = (char *) calloc((size_t) lenhbo, sizeof(char));
	if(buf == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	fread(buf, lenhbo, 1, fp);
	if(feof(fp) || ferror(fp)) {
		perrorxcf(path, fp, "could not read layer name data");
		if(buf) {
			free(buf);
		}

		return NULL;
	}

	if(buf[lenhbo - 1] != '\0') {
		perrorxcf(path, fp, "layer name data does not end with NUL");
		if(buf) {
			free(buf);
		}

		return NULL;
	}

	char *_tmp_1;
	_tmp_1 = buf;
	buf = NULL;
	return (char *) _tmp_1;
}

XcfLayer * read_xcf_layer_from_fp(FILE * fp, long long offsethbo, XcfPrecision precision, const char * path)
{
	XcfProperty *prp;
	long bakpos = ftell(fp);

	fseek(fp, offsethbo, SEEK_SET);
	if(feof(fp) || ferror(fp)) {
		perrorxcf(path, fp, "could not seek to layer offset %lld", offsethbo);
		return NULL;
	}

	_ngg_tuple_read_dimentions_from_xcf_fp _ngg_tmp_1 = read_dimentions_from_xcf_fp(fp, path);
	_Bool ok_layerdim = _ngg_tmp_1.m2;
	int height_layer = _ngg_tmp_1.m1;
	int width_layer = _ngg_tmp_1.m0;
	if(!ok_layerdim) {
		perrorxcf(path, fp, "could not read layer dimentions");
		return NULL;
	}

	logxcf(path, "read layer dimentions: %d x %d", width_layer, height_layer);

	unsigned int typebe = 0u;

	fread(&typebe, sizeof(typebe), 1, fp);
	if(feof(fp) || ferror(fp)) {
		perrorxcf(path, fp, "could not read layer type");
		return NULL;
	}

	char * layernam = read_string_pointers_from_xcf_fp(fp, path);
	if(layernam) {
		char * layernamnn;
		layernamnn = (char *) layernam;
		logxcf(path, "read layer name: %s", layernamnn);
	} else {
		perrorxcf(path, fp, "could not read layer name");
		if(layernam) {
			free(layernam);
		}

		return NULL;
	}

	XcfProperty *propshead = read_raw_xcf_properties_from_fp(fp, path);
	if(!propshead) {
		perrorxcf(path, fp, "could not read layer properties");
		if(layernam) {
			free(layernam);
		}

		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		return NULL;
	}

	_ngg_tuple_read_xcf_pointer_from_fp _ngg_tmp_2 = read_xcf_pointer_from_fp(fp, path);
	_Bool ok_hierptr = _ngg_tmp_2.m1;
	long long hierarchyptr = _ngg_tmp_2.m0;
	if(!ok_hierptr) {
		perrorxcf(path, fp, "could not read pointer to hierarchy data");
		if(layernam) {
			free(layernam);
		}

		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		return NULL;
	}

	_ngg_tuple_read_xcf_pointer_from_fp _ngg_tmp_3 = read_xcf_pointer_from_fp(fp, path);
	_Bool ok_maskptr = _ngg_tmp_3.m1;
	long long maskptr = _ngg_tmp_3.m0;
	if(!ok_maskptr) {
		perrorxcf(path, fp, "could not read layer mask pointer");
		if(layernam) {
			free(layernam);
		}

		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		return NULL;
	}

	if(maskptr != 0) {
		logxcf(path, "ignoring layer mask");
	}

	fseek(fp, hierarchyptr, SEEK_SET);
	if(feof(fp) || ferror(fp)) {
		perrorxcf(path, fp, "could not seek to hierarchy offset %lld", hierarchyptr);
		if(layernam) {
			free(layernam);
		}

		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		return NULL;
	}

	_ngg_tuple_read_dimentions_from_xcf_fp _ngg_tmp_4 = read_dimentions_from_xcf_fp(fp, path);
	_Bool ok_hierdim = _ngg_tmp_4.m2;
	int height_hier = _ngg_tmp_4.m1;
	int width_hier = _ngg_tmp_4.m0;
	if(!ok_hierdim) {
		perrorxcf(path, fp, "could not read hierarchy dimentions");
		if(layernam) {
			free(layernam);
		}

		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		return NULL;
	}

	unsigned int bppbe = 0u;

	fread(&bppbe, sizeof(bppbe), 1, fp);
	if(feof(fp) || ferror(fp)) {
		perrorxcf(path, fp, "could not read bpp");
		if(layernam) {
			free(layernam);
		}

		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		return NULL;
	}

	XcfPointerVector *levelptrs = read_xcf_pointers_from_fp(fp, path, "level");
	if(!levelptrs) {
		if(layernam) {
			free(layernam);
		}

		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		if(levelptrs) {
			xcf_pointer_vector_destruct((XcfPointerVector *) levelptrs);
			free(levelptrs);
		}

		return NULL;
	}

	assert(levelptrs);
	if(ftell(fp) != xcf_pointer_vector_get_item(levelptrs, 0)) {
		perrorxcf(path, fp, "unexpected level pointer");
		if(layernam) {
			free(layernam);
		}

		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		if(levelptrs) {
			xcf_pointer_vector_destruct((XcfPointerVector *) levelptrs);
			free(levelptrs);
		}

		return NULL;
	}

	_ngg_tuple_read_dimentions_from_xcf_fp _ngg_tmp_5 = read_dimentions_from_xcf_fp(fp, path);
	_Bool ok_leveldim = _ngg_tmp_5.m2;
	int height_level = _ngg_tmp_5.m1;
	int width_level = _ngg_tmp_5.m0;
	if(!ok_leveldim) {
		perrorxcf(path, fp, "could not read level dimentions");
		if(layernam) {
			free(layernam);
		}

		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		if(levelptrs) {
			xcf_pointer_vector_destruct((XcfPointerVector *) levelptrs);
			free(levelptrs);
		}

		return NULL;
	}

	if((((width_layer != width_hier) || (width_layer != width_level)) || (height_layer != height_hier)) || (height_layer != height_level)) {
		perrorxcf(path, fp, "mismatch between layer, hierarchy, and level dimentions");

		if(layernam) {
			free(layernam);
		}

		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		if(levelptrs) {
			xcf_pointer_vector_destruct((XcfPointerVector *) levelptrs);
			free(levelptrs);
		}

		return NULL;
	}

	XcfPointerVector *tileptrs = read_xcf_pointers_from_fp(fp, path, "tile");
	if(!tileptrs) {
		if(layernam) {
			free(layernam);
		}

		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		if(levelptrs) {
			xcf_pointer_vector_destruct((XcfPointerVector *) levelptrs);
			free(levelptrs);
		}

		if(tileptrs) {
			xcf_pointer_vector_destruct((XcfPointerVector *) tileptrs);
			free(tileptrs);
		}

		return NULL;
	}

	assert(tileptrs);
	XcfTileVector *tiles = read_xcf_tiles_from_fp(fp, tileptrs, width_level, height_level, precision, (int) ntohl(bppbe), path);
	if(!tiles) {
		if(layernam) {
			free(layernam);
		}

		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		if(levelptrs) {
			xcf_pointer_vector_destruct((XcfPointerVector *) levelptrs);
			free(levelptrs);
		}

		if(tileptrs) {
			xcf_pointer_vector_destruct((XcfPointerVector *) tileptrs);
			free(tileptrs);
		}

		return NULL;
	}

	fseek(fp, bakpos, SEEK_SET);
	if(feof(fp) || ferror(fp)) {
		perrorxcf(path, fp, "could not seek to %ld after reading layer", bakpos);
		if(layernam) {
			free(layernam);
		}

		if(propshead) {
			xcf_property_destruct((XcfProperty *) propshead);
			free(propshead);
		}

		if(levelptrs) {
			xcf_pointer_vector_destruct((XcfPointerVector *) levelptrs);
			free(levelptrs);
		}

		if(tileptrs) {
			xcf_pointer_vector_destruct((XcfPointerVector *) tileptrs);
			free(tileptrs);
		}

		return NULL;
	}

	char * _tmp_1;
	_tmp_1 = layernam;
	layernam = NULL;
	assert(_tmp_1);
	assert(tiles);
	XcfLayer *layer = (XcfLayer *) malloc(sizeof(XcfLayer));
	if(layer == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	xcf_layer_construct(layer, _tmp_1, width_layer, height_layer, tiles);

	assert(propshead);
	XcfProperty *propsheadnn = propshead;
	prp = xcf_property_get_next(propsheadnn);
	while(prp) {
		switch(prp->id) {
		case XCF_PROP_FLOAT_OPACITY:
		{
			xcf_layer_set_alpha(layer, prp->value.fval);
			break;
		}
		case XCF_PROP_VISIBLE:
		{
			xcf_layer_set_visible(layer, prp->value.bval);
			break;
		}
		default:
		{
			break;
		}
		}

		prp = (XcfProperty *) xcf_property_get_next(prp);
	}

	if(layernam) {
		free(layernam);
	}

	if(propshead) {
		xcf_property_destruct((XcfProperty *) propshead);
		free(propshead);
	}

	if(levelptrs) {
		xcf_pointer_vector_destruct((XcfPointerVector *) levelptrs);
		free(levelptrs);
	}

	if(tileptrs) {
		xcf_pointer_vector_destruct((XcfPointerVector *) tileptrs);
		free(tileptrs);
	}

	return layer;
}

_ngg_tuple_read_xcf_pointer_from_fp read_xcf_pointer_from_fp(FILE * fp, const char * path)
{
	long long ptr = 0;

	fread(&ptr, sizeof(ptr), 1, fp);
	if(feof(fp) || ferror(fp)) {
		return (_ngg_tuple_read_xcf_pointer_from_fp){0, false};
	}

	return (_ngg_tuple_read_xcf_pointer_from_fp){(long long) GINT64_FROM_BE(ptr), true};
}

XcfPointerVector * read_xcf_pointers_from_fp(FILE * fp, const char * path, const char * purpose)
{
	XcfPointerVector *ptrv = (XcfPointerVector *) malloc(sizeof(XcfPointerVector));
	if(ptrv == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	xcf_pointer_vector_construct(ptrv);

	while(true) {
		_ngg_tuple_read_xcf_pointer_from_fp _ngg_tmp_6 = read_xcf_pointer_from_fp(fp, path);
		_Bool ok = _ngg_tmp_6.m1;
		long long ptr = _ngg_tmp_6.m0;
		if(!ok) {
			perrorxcf(path, fp, "could not read pointer (%s)", purpose);
			if(ptrv) {
				xcf_pointer_vector_destruct(ptrv);
				free(ptrv);
			}

			return NULL;
		}

		if(ptr == 0) {
			break;
		}

		if(ptr > get_max_pointer(fp)) {
			perrorxcf(path, fp, "invalid pointer %lld (%s)", ptr, purpose);
			if(ptrv) {
				xcf_pointer_vector_destruct(ptrv);
				free(ptrv);
			}

			return NULL;
		}

		xcf_pointer_vector_append(ptrv, ptr);
	}

	XcfPointerVector *_tmp_1;
	_tmp_1 = ptrv;
	ptrv = NULL;
	return _tmp_1;
}

XcfTile * read_xcf_tile_from_fp(FILE * fp, long long offsethbo, int width, int height, XcfPrecision precision, int bpp, int nchannel, const char * path)
{
	int channel;
	long bakpos = ftell(fp);

	fseek(fp, offsethbo, SEEK_SET);
	if(feof(fp) || ferror(fp)) {
		perrorxcf(path, fp, "could not seek to tile offset %lld", offsethbo);
		return NULL;
	}

	XcfTile *tile = (XcfTile *) malloc(sizeof(XcfTile));
	if(tile == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	xcf_tile_construct(tile, width, height);

	unsigned char *buf = (unsigned char *) calloc((size_t) 4096, sizeof(unsigned char));
	if(buf == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	int npixel = width * height;

	for(channel = 0; channel < nchannel; channel += 1) {
		int byt;
		logxcf(path, "reading channel %d", channel);

		for(byt = 0; byt < (bpp / nchannel); byt += 1) {
			int y;
			logxcf(path, "reading byte %d", byt);

			int nbyte_read = 0;
			while(nbyte_read < npixel) {
				unsigned char rleopc = (unsigned char) (0u);
				fread(&rleopc, sizeof(unsigned char), 1, fp);
				if(feof(fp) || ferror(fp)) {
					perrorxcf(path, fp, "could not read RLE opcode");
					if(tile) {
						xcf_tile_destruct(tile);
						free(tile);
					}

					if(buf) {
						free(buf);
					}

					return NULL;
				}

				int _tmp_2;
				if(rleopc == (127u)) {
					_tmp_2 = 3;
				} else {
					_tmp_2 = ((rleopc == (128u)) ? 2 : 0);
				}

				int rlehdrlen = ((rleopc <= (126u)) ? 1 : _tmp_2);

				unsigned char rlehdr[3];
				if(rlehdrlen) {
					fread(rlehdr, rlehdrlen * sizeof(unsigned char), 1, fp);
					if(feof(fp) || ferror(fp)) {
						perrorxcf(path, fp, "could not read RLE header");
						if(tile) {
							xcf_tile_destruct(tile);
							free(tile);
						}

						if(buf) {
							free(buf);
						}

						return NULL;
					}
				}

				int count = 0;

				if(rleopc <= (127u)) {
					count = (int) (rleopc + (1u));
					unsigned char value = rlehdr[0];

					if(rleopc == (127u)) {
						count = (int) ((rlehdr[0] * (256u)) + rlehdr[1]);
						value = rlehdr[2];
					}

					if((nbyte_read + count) > 4096) {
						perrorxcf(path, fp, "RLE overflow (opcode = %u, bytes read = %d, available = %d)", rleopc, nbyte_read, count);
						if(tile) {
							xcf_tile_destruct(tile);
							free(tile);
						}

						if(buf) {
							free(buf);
						}

						return NULL;
					}

					memset(buf + nbyte_read, (int) value, (size_t) count);
				} else {
					unsigned int _tmp_3;
					if(rleopc == (128u)) {
						_tmp_3 = (rlehdr[0] * (256u)) + rlehdr[1];
					} else {
						_tmp_3 = (256u) - rleopc;
					}

					count = (int) _tmp_3;

					if((nbyte_read + count) > 4096) {
						perrorxcf(path, fp, "RLE overflow (opcode = %u, bytes read = %d, available = %d)", rleopc, nbyte_read, count);
						if(tile) {
							xcf_tile_destruct(tile);
							free(tile);
						}

						if(buf) {
							free(buf);
						}

						return NULL;
					}

					fread(buf + nbyte_read, (size_t) count, 1, fp);
					if(feof(fp) || ferror(fp)) {
						perrorxcf(path, fp, "could not read tile data");
						if(tile) {
							xcf_tile_destruct(tile);
							free(tile);
						}

						if(buf) {
							free(buf);
						}

						return NULL;
					}
				}

				nbyte_read += count;
			}

			assert(nbyte_read <= 4096); /* xcf-read.ngg:694 */

			for(y = 0; y < height; y += 1) {
				int x;
				for(x = 0; x < width; x += 1) {
					xcf_tile_set_byte(tile, x, y, channel, byt, buf[(y * width) + x]);
				}
			}
		}
	}

	fseek(fp, bakpos, SEEK_SET);
	if(feof(fp) || ferror(fp)) {
		perrorxcf(path, fp, "could not seek to %ld after reading tile", bakpos);
		if(tile) {
			xcf_tile_destruct(tile);
			free(tile);
		}

		if(buf) {
			free(buf);
		}

		return NULL;
	}

	switch(precision.bits) {
	case 8:
	{
		if(precision.linear) {
			int y;
			for(y = 0; y < height; y += 1) {
				int x;
				for(x = 0; x < width; x += 1) {
					xcf_tile_scale_pixel_color_components_from_linear_byte(tile, x, y);
				}
			}
		} else {
			int y;
			for(y = 0; y < height; y += 1) {
				int x;
				for(x = 0; x < width; x += 1) {
					xcf_tile_scale_pixel_color_components_from_gamma_byte(tile, x, y);
				}
			}
		}

		if(nchannel == 4) {
			int y;
			for(y = 0; y < height; y += 1) {
				int x;
				for(x = 0; x < width; x += 1) {
					xcf_tile_scale_pixel_alpha_component_from_byte(tile, x, y);
				}
			}
		}
		break;
	}
	case 32:
	{
		if(!precision.linear) {
			perrorxcf(path, fp, "unsupported precision: %u-bit gamma", precision.bits);
			if(tile) {
				xcf_tile_destruct(tile);
				free(tile);
			}

			if(buf) {
				free(buf);
			}

			return NULL;
		}
		break;
	}
	}

	if(nchannel == 3) {
		int y;
		for(y = 0; y < height; y += 1) {
			int x;
			for(x = 0; x < width; x += 1) {
				xcf_tile_set_pixel_component(tile, x, y, 3, 1);
			}
		}
	} else {
		int y;
		for(y = 0; y < height; y += 1) {
			int x;
			for(x = 0; x < width; x += 1) {
				for(channel = 0; channel < 3; channel += 1) {
					xcf_tile_set_pixel_component(tile, x, y, channel, xcf_tile_get_pixel_component(tile, x, y, 3) * xcf_tile_get_pixel_component(tile, x, y, channel));
				}
			}
		}
	}

	xcf_tile_finish(tile);

	XcfTile *_tmp_1;
	_tmp_1 = tile;
	tile = NULL;
	if(buf) {
		free(buf);
	}

	return _tmp_1;
}

XcfTileVector * read_xcf_tiles_from_fp(FILE * fp, XcfPointerVector *tileptrs, int width, int height, XcfPrecision precision, int bpp, const char * path)
{
	int y;
	int ntileh = (int) ceil(width / 64.0);

	int ntilev = (int) ceil(height / 64.0);
	int ntile = ntileh * ntilev;

	logxcf(path, "expecting %d tiles", ntile);

	if(xcf_pointer_vector_get_count(tileptrs) != ntile) {
		logxcf(path, "unexpected number of tile pointers: %d", xcf_pointer_vector_get_count(tileptrs));
		return NULL;
	}

	int nchannel = 0;

	switch(bpp) {
	case 3:
	case 6:
	{
		nchannel = 3;
		break;
	}
	case 4:
	case 16:
	{
		nchannel = 4;
		break;
	}
	default:
	{
		perrorxcf(path, fp, "unsupported bpp: %d", bpp);
		return NULL;
		break;
	}
	}

	XcfTileVector *tiles = (XcfTileVector *) malloc(sizeof(XcfTileVector));
	if(tiles == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	xcf_tile_vector_construct(tiles);

	for(y = 0; y < ntilev; y += 1) {
		int x;
		int _tmp_2;
		if((y == (ntilev - 1)) && (height % 64)) {
			_tmp_2 = height % 64;
		} else {
			_tmp_2 = 64;
		}

		int th = _tmp_2;

		for(x = 0; x < ntileh; x += 1) {
			int _tmp_3;
			if((x == (ntileh - 1)) && (width % 64)) {
				_tmp_3 = width % 64;
			} else {
				_tmp_3 = 64;
			}

			int tw = _tmp_3;

			int tilei = (y * ntileh) + x;

			logxcf(path, "reading tile %d (%d x %d)", tilei, tw, th);

			XcfTile *tile = read_xcf_tile_from_fp(fp, xcf_pointer_vector_get_item(tileptrs, tilei), tw, th, precision, bpp, nchannel, path);
			if(tile) {
				XcfTile *tilenn;
				tilenn = (XcfTile *) tile;
				xcf_tile_vector_append(tiles, tilenn);
			} else {
				if(tiles) {
					xcf_tile_vector_destruct(tiles);
					free(tiles);
				}

				return NULL;
			}
		}
	}

	XcfTileVector *_tmp_1;
	_tmp_1 = tiles;
	tiles = NULL;
	return _tmp_1;
}

XcfProperty * read_raw_xcf_properties_from_fp(FILE * fp, const char * path)
{
	XcfPropertyValue nilval;
	XcfProperty *head = (XcfProperty *) malloc(sizeof(XcfProperty));
	if(head == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	xcf_property_construct(head, 0u, 0u, nilval);
	XcfProperty *tail = head;

	char *buf = (char *) calloc((size_t) sizeof(unsigned int), sizeof(char));
	if(buf == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	_Bool had_unknown_ppty = false;

	while(true) {
		unsigned int idbe = 0u;
		unsigned int lenbe = idbe;

		fread(&idbe, sizeof(idbe), 1, fp);
		if(feof(fp) || ferror(fp)) {
			perrorxcf(path, fp, "could not read property id");
			if(head) {
				xcf_property_destruct(head);
				free(head);
			}

			if(buf) {
				free(buf);
			}

			return NULL;
		}

		unsigned int idhbo = ntohl(idbe);

		fread(&lenbe, sizeof(lenbe), 1, fp);
		if(feof(fp) || ferror(fp)) {
			perrorxcf(path, fp, "could not read property length (id: %u)", idhbo);
			if(head) {
				xcf_property_destruct(head);
				free(head);
			}

			if(buf) {
				free(buf);
			}

			return NULL;
		}

		unsigned int lenhbo = ntohl(lenbe);
		if(lenhbo > MAX_PPTYLEN) {
			perrorxcf(path, fp, "property with length > %u (id: %u)", MAX_PPTYLEN, idhbo);
			if(head) {
				xcf_property_destruct(head);
				free(head);
			}

			if(buf) {
				free(buf);
			}

			return NULL;
		}

		if(lenhbo > 0) {
			buf = realloc(buf, lenhbo * sizeof(buf[0]));
			if(buf == NULL) { perror(NULL); exit(EXIT_FAILURE); }

		}

		fread(buf, lenhbo, 1, fp);
		if(feof(fp) || ferror(fp)) {
			perrorxcf(path, fp, "could not read property value (id: %u)", idhbo);
			if(head) {
				xcf_property_destruct(head);
				free(head);
			}

			if(buf) {
				free(buf);
			}

			return NULL;
		}

		if(idhbo == (0u)) {
			break;
		}

		XcfPropertyValue value;

		switch(idhbo) {
		case XCF_PROP_COMPRESSION:
		case XCF_PROP_MODE:
		{
			memcpy(&value.uval, (const char *) buf, sizeof(value.uval));
			value.uval = ntohl(value.uval);
			break;
		}
		case XCF_PROP_APPLY_MASK:
		case XCF_PROP_VISIBLE:
		{
			unsigned int rawval = 0u;
			memcpy(&rawval, (const char *) buf, sizeof(rawval));
			value.bval = (_Bool) rawval;
			break;
		}
		case XCF_PROP_FLOAT_OPACITY:
		{
			unsigned int rawval = 0u;
			memcpy(&rawval, (const char *) buf, sizeof(rawval));
			rawval = ntohl(rawval);
			memcpy(&value.fval, (const char *) (&rawval), sizeof(rawval));
			break;
		}
		case XCF_PROP_OFFSETS:
		{
			unsigned int rawx = 0u;
			unsigned int rawy = rawx;
			memcpy(&rawx, (const char *) buf, sizeof(rawx));
			memcpy(&rawy, (const char *) (buf + sizeof(rawx)), sizeof(rawy));
			value.ival2[0] = (int) ntohl(rawx);
			value.ival2[1] = (int) ntohl(rawy);
			break;
		}
		case XCF_PROP_ACTIVE_LAYER:
		case XCF_PROP_BLEND_SPACE:
		case XCF_PROP_COLOR_TAG:
		case XCF_PROP_COMPOSITE_MODE:
		case XCF_PROP_COMPOSITE_SPACE:
		case XCF_PROP_EDIT_MASK:
		case XCF_PROP_LINKED:
		case XCF_PROP_LOCK_ALPHA:
		case XCF_PROP_LOCK_CONTENT:
		case XCF_PROP_LOCK_POSITION:
		case XCF_PROP_OPACITY:
		case XCF_PROP_PARASITES:
		case XCF_PROP_RESOLUTION:
		case XCF_PROP_SHOW_MASK:
		case XCF_PROP_TATTOO:
		case XCF_PROP_UNIT:
		{
			logxcf(path, "ignoring property: %u", idhbo);
			continue;
			break;
		}
		default:
		{
			perrorxcf(path, fp, "unknown property: %u", idhbo);

			had_unknown_ppty = true;
			continue;
			break;
		}
		}

		XcfProperty *ppty = (XcfProperty *) malloc(sizeof(XcfProperty));
		if(ppty == NULL) {
			perror(NULL);
			exit(EXIT_FAILURE);
		}

		xcf_property_construct(ppty, idhbo, lenhbo, value);
		XcfProperty *_ngg_tmp_10 = ppty;
		if(tail->next) {
			xcf_property_destruct((XcfProperty *) tail->next);
			free(tail->next);
		}

		tail->next = _ngg_tmp_10;
		tail = ppty;
	}

	if(had_unknown_ppty) {
		if(head) {
			xcf_property_destruct(head);
			free(head);
		}

		if(buf) {
			free(buf);
		}

		return NULL;
	}

	XcfProperty *_tmp_1;
	_tmp_1 = head;
	head = NULL;
	if(buf) {
		free(buf);
	}

	return _tmp_1;
}

_ngg_tuple_decode_xcf_precision decode_xcf_precision(unsigned int precision)
{
	XcfPrecision pr = xcf_precision_default();

	switch(precision) {
	case 100:
	case 150:
	{
		pr.bits = 8;
		break;
	}
	case 600:
	case 650:
	{
		pr.bits = 32;
		break;
	}
	default:
	{
		return (_ngg_tuple_decode_xcf_precision){pr, false};
		break;
	}
	}

	pr.linear = (0u) == (precision % (100u));

	return (_ngg_tuple_decode_xcf_precision){pr, true};
}

unsigned char get_byte_in_float(float f, int i)
{
	return ((unsigned char *) (&f))[i];
}

float set_byte_in_float(float f, int i, unsigned char byt)
{
	((unsigned char *) (&f))[i] = byt;
	return f;
}

XcfPrecision xcf_precision_default()
{
	XcfPrecision s;
	s.bits = 0;
	s.linear = false;
	return s;
}

_ngg_tuple_read_dimentions_from_xcf_fp _ngg_tuple_read_dimentions_from_xcf_fp_default()
{
	_ngg_tuple_read_dimentions_from_xcf_fp s;
	s.m0 = 0;
	s.m1 = 0;
	s.m2 = false;
	return s;
}

_ngg_tuple_read_xcf_pointer_from_fp _ngg_tuple_read_xcf_pointer_from_fp_default()
{
	_ngg_tuple_read_xcf_pointer_from_fp s;
	s.m0 = 0;
	s.m1 = false;
	return s;
}

_ngg_tuple_decode_xcf_precision _ngg_tuple_decode_xcf_precision_default()
{
	_ngg_tuple_decode_xcf_precision s;
	s.m0 = xcf_precision_default();
	s.m1 = false;
	return s;
}
