/* xcf-write.c
 * This file is part of Vara digital painting program
 * Copyright 2021, 2023, 2024, 2025 Nandakumar Edamana
 * Distributed under GNU General Public License, version 3
 */ 

#include <assert.h>
#include <stdarg.h>
#include <stdbool.h>

#include "xcf-write.h"


#include <arpa/inet.h>

#include <glib.h>

void pointer_replacement_vector__resize(PointerReplacementVector *this, int newcount)
{
	if(newcount > this->alloccount) {
		int alloccountbak = this->alloccount;

		this->alloccount = 2 * this->alloccount;

		assert(this->alloccount > alloccountbak); /* vector.ngg:17 */

		assert(newcount <= this->alloccount); /* vector.ngg:18 */

		this->arr = realloc(this->arr, this->alloccount * sizeof(this->arr[0]));
		if(this->arr == NULL) { perror(NULL); exit(EXIT_FAILURE); }

	}

	this->count = newcount;
}

void pointer_replacement_vector_append(PointerReplacementVector *this, PointerReplacement newitem)
{
	int newcount = 1 + this->count;

	pointer_replacement_vector__resize(this, newcount);

	this->arr[newcount - 1] = newitem;
}

void pointer_replacement_vector_clear(PointerReplacementVector *this)
{
	this->count = 0;
	this->alloccount = 8;
	this->arr = realloc(this->arr, this->alloccount * sizeof(this->arr[0]));
	if(this->arr == NULL) { perror(NULL); exit(EXIT_FAILURE); }

}

PointerReplacement pointer_replacement_vector_get_item(PointerReplacementVector *this, int index)
{
	assert(index < this->count); /* vector.ngg:43 */

	return this->arr[index];
}

PointerReplacement pointer_replacement_vector_pop(PointerReplacementVector *this)
{
	assert(this->count > 0); /* vector.ngg:49 */
	PointerReplacement r = pointer_replacement_vector_get_item(this, this->count - 1);
	this->count = this->count - 1;

	return r;
}

void pointer_replacement_vector_set_item(PointerReplacementVector *this, int index, PointerReplacement itm)
{
	assert(index < this->count); /* vector.ngg:62 */

	this->arr[index] = itm;
}

int pointer_replacement_vector_get_count(PointerReplacementVector *this)
{
	return this->count;
}

_Bool pointer_replacement_vector_is_empty(PointerReplacementVector *this)
{
	return 0 == this->count;
}

void pointer_replacement_vector_destruct(PointerReplacementVector *this)
{
	if(this->arr) {
		free(this->arr);
	}
}

void pointer_replacement_vector_construct(PointerReplacementVector *this)
{
	this->alloccount = 8;
	PointerReplacement *_tmp_1 = (PointerReplacement *) calloc((size_t) 8, sizeof(PointerReplacement));
	if(_tmp_1 == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	this->arr = _tmp_1;
	this->count = 0;
}

void pointer_replacement_context_apply(PointerReplacementContext *this, FILE * fp)
{
	int i;
	PointerReplacement repl;
	long locbak = ftell(fp);

	for(i = 0; i < pointer_replacement_vector_get_count(this->prvec); i += 1) {
		repl = pointer_replacement_vector_get_item(this->prvec, i);
		assert(repl.ptrloc >= 0); /* xcf-write.ngg:38 */

		if(0 != fseek(fp, repl.ptrloc, SEEK_SET)) {
			return;
		}

		nanxcf_write_absptrbe(fp, (long long) GINT64_TO_BE((long long) pointer_replacement_vector_get_item(this->prvec, i).ptrval));
	}

	if(0 != fseek(fp, locbak, SEEK_SET)) {
		return;
	}
}

void pointer_replacement_context_put_placeholder(PointerReplacementContext *this, FILE * fp)
{
	pointer_replacement_vector_append(this->prvec, (PointerReplacement) { ftell(fp), 0 });
	nanxcf_write_absptrbe(fp, 0);
}

void pointer_replacement_context_set_ptrval(PointerReplacementContext *this, int idx, long val)
{
	PointerReplacement repl = pointer_replacement_vector_get_item(this->prvec, idx);
	repl.ptrval = val;
	pointer_replacement_vector_set_item(this->prvec, idx, repl);
}

void pointer_replacement_context_destruct(PointerReplacementContext *this)
{
	if(this->prvec) {
		pointer_replacement_vector_destruct(this->prvec);
		free(this->prvec);
	}
}

void pointer_replacement_context_construct(PointerReplacementContext *this)
{
	PointerReplacementVector *_tmp_1 = (PointerReplacementVector *) malloc(sizeof(PointerReplacementVector));
	if(_tmp_1 == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	pointer_replacement_vector_construct(_tmp_1);
	this->prvec = _tmp_1;
}

unsigned int ZERO;

int write_xcf_from_image_document(const char * path, ImageDocument *doc)
{
	int _ngg_tmp_2;
	int _ngg_tmp_1;
	int idxidx;
	ZERO = htonl(0u);

	PointerReplacementContext *layerptrs = (PointerReplacementContext *) malloc(sizeof(PointerReplacementContext));
	if(layerptrs == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	pointer_replacement_context_construct(layerptrs);

	FILE * fpnul = fopen(path, "wb");
	if(!fpnul) {
		if(layerptrs) {
			pointer_replacement_context_destruct(layerptrs);
			free(layerptrs);
		}

		return -1;
	}

	FILE * fp = (FILE *) fpnul;

	XcfHeader hdr = xcf_header_default();

	memcpy(hdr.magic, "gimp xcf ", 9u);

	memcpy(hdr.version, "v011\0", 5u);

	hdr.width = htonl((unsigned int) doc->width);
	hdr.height = htonl((unsigned int) doc->height);

	hdr.base_type = htonl(0u);

	hdr.precision = htonl(600u);

	fwrite(&hdr, sizeof(hdr), 1u, fp);

	nanxcf_write_prop_compression_rle(fp);
	nanxcf_write_prop_end(fp);
	for(_ngg_tmp_1 = 0; _ngg_tmp_1 < intvector_get_count(doc->layerstack); _ngg_tmp_1 += 1) {
		idxidx = intvector_get_item(doc->layerstack, _ngg_tmp_1);
		pointer_replacement_context_put_placeholder(layerptrs, fp);
	}

	nanxcf_write_absptrbe(fp, 0);
	nanxcf_write_absptrbe(fp, 0);
	nanxcf_write_absptrbe(fp, 0);
	for(_ngg_tmp_2 = 0; _ngg_tmp_2 < intvector_get_count(doc->layerstack); _ngg_tmp_2 += 1) {
		idxidx = intvector_get_item(doc->layerstack, _ngg_tmp_2);
		pointer_replacement_context_set_ptrval(layerptrs, (intvector_get_count(doc->layerstack) - 1) - idxidx, ftell(fp));
		int idx = intvector_get_item(doc->layerstack, idxidx);
		nanxcf_write_layer(fp, layer_owning_vector_get_item(doc->layers, idx));
		if(ferror(fp)) {
			break;
		}
	}

	pointer_replacement_context_apply(layerptrs, fp);
	int ferr = ferror(fp);

	fclose(fp);

	if(layerptrs) {
		pointer_replacement_context_destruct(layerptrs);
		free(layerptrs);
	}

	return (int) ferr;
}

void nanxcf_write_absptrbe(FILE * fp, long long ptr)
{
	fwrite(&ptr, sizeof(ptr), 1u, fp);
}

void nanxcf_write_byte(FILE * fp, unsigned char b)
{
	fwrite(&b, sizeof(b), 1u, fp);
}

void nanxcf_write_float(FILE * fp, float x)
{
	unsigned int u = 0u;
	assert(sizeof(unsigned int) == sizeof(float)); /* xcf-write.ngg:152 */
	memcpy((char *) (&u), (const char *) (&x), sizeof(unsigned int));
	nanxcf_write_uint(fp, u);
}

void nanxcf_write_int(FILE * fp, int n)
{
	unsigned int finaln = 0u;
	memcpy((char *) (&finaln), (const char *) (&n), sizeof(finaln));
	nanxcf_write_uint(fp, finaln);
}

void nanxcf_write_layer(FILE * fp, Layer *layer)
{
	int ytile;
	int j;
	int i;
	unsigned int finalw = htonl((unsigned int) layer->img->width);
	unsigned int finalh = htonl((unsigned int) layer->img->height);

	fwrite(&finalw, sizeof(finalw), 1u, fp);
	fwrite(&finalh, sizeof(finalh), 1u, fp);

	nanxcf_write_uint(fp, 1u);
	nanxcf_write_string(fp, layer->name);
	nanxcf_write_prop_layer_mode(fp, 28u);
	nanxcf_write_prop_offsets_0_0(fp);
	nanxcf_write_prop_visible(fp, true);
	nanxcf_write_prop_float_opacity(fp, layer->alpha);
	nanxcf_write_prop_apply_layer_mask(fp, false);
	nanxcf_write_prop_end(fp);

	nanxcf_write_relptr(fp, (long long) (3 * sizeof(long long)));

	nanxcf_write_absptrbe(fp, 0);
	nanxcf_write_absptrbe(fp, 0);
	fwrite(&finalw, sizeof(finalw), 1u, fp);
	fwrite(&finalh, sizeof(finalh), 1u, fp);

	int nchannel = 4;

	unsigned int bpp = htonl((unsigned int) (nchannel * sizeof(float)));
	fwrite(&bpp, sizeof(bpp), 1u, fp);

	nanxcf_write_relptr(fp, (long long) (2 * sizeof(long long)));

	nanxcf_write_absptrbe(fp, 0);
	fwrite(&finalw, sizeof(finalw), 1u, fp);
	fwrite(&finalh, sizeof(finalh), 1u, fp);

	int hceil = (int) ceil(layer->img->height / 64.0);
	int wceil = (int) ceil(layer->img->width / 64.0);
	int ntile = hceil * wceil;

	PointerReplacementContext *tileptrs = (PointerReplacementContext *) malloc(sizeof(PointerReplacementContext));
	if(tileptrs == NULL) {
		perror(NULL);
		exit(EXIT_FAILURE);
	}

	pointer_replacement_context_construct(tileptrs);

	for(i = 0; i < ntile; i += 1) {
		pointer_replacement_context_put_placeholder(tileptrs, fp);
	}

	nanxcf_write_absptrbe(fp, 0);
	Image *imgstraight = image_Clone(layer->img);
	for(j = 0; j < imgstraight->height; j += 1) {
		for(i = 0; i < imgstraight->width; i += 1) {
			int c;
			LinearPixel px = nan_img_get_linear_pixel(imgstraight, i, j);
			for(c = 0; c < 3; c += 1) {
				px.rgb[c] = px.rgb[c] / fnonzero_or_one(px.rgb[3]);
			}

			nan_img_set_linear_pixel(imgstraight, i, j, px);
		}
	}

	int tileidx = 0;

	int ystart = 0;

	for(ytile = 0; ytile < hceil; ytile += 1) {
		int xtile;
		int xstart = 0;

		int npixely = imin2(imgstraight->height, (ytile + 1) * 64) - (ytile * 64);

		for(xtile = 0; xtile < wceil; xtile += 1) {
			int channel;
			pointer_replacement_context_set_ptrval(tileptrs, tileidx, ftell(fp));

			int npixelx = imin2(imgstraight->width, (xtile + 1) * 64) - (xtile * 64);
			int npixel = npixelx * npixely;
			int p = npixel / 256;
			int q = npixel % 256;

			for(channel = 0; channel < nchannel; channel += 1) {
				int byt;
				for(byt = 0; byt < sizeof(float); byt += 1) {
					int y;
					nanxcf_write_byte(fp, (unsigned char) 128);
					nanxcf_write_byte(fp, (unsigned char) p);
					nanxcf_write_byte(fp, (unsigned char) q);

					for(y = ystart; y < imin2(ystart + 64, imgstraight->height); y += 1) {
						int x;
						for(x = xstart; x < imin2(xstart + 64, imgstraight->width); x += 1) {
							LinearPixel px = nan_img_get_linear_pixel(imgstraight, x, y);
							float clr = px.rgb[channel];
							nanxcf_write_byte(fp, get_byte_from_float(clr, byt));
						}
					}
				}
			}

			xstart += 64;
			tileidx += 1;
		}

		ystart += 64;
	}

	pointer_replacement_context_apply(tileptrs, fp);
	if(tileptrs) {
		pointer_replacement_context_destruct(tileptrs);
		free(tileptrs);
	}

	if(imgstraight) {
		image_destruct(imgstraight);
		free(imgstraight);
	}
}

void nanxcf_write_prop_apply_layer_mask(FILE * fp, _Bool apply)
{
	nanxcf_write_uint_prop(fp, XCF_PROP_APPLY_MASK, (unsigned int) apply);
}

void nanxcf_write_prop_blend_space(FILE * fp, int space)
{
	nanxcf_write_int_prop(fp, XCF_PROP_BLEND_SPACE, space);
}

void nanxcf_write_prop_composite_mode(FILE * fp, int mode)
{
	nanxcf_write_int_prop(fp, XCF_PROP_COMPOSITE_MODE, mode);
}

void nanxcf_write_prop_composite_space(FILE * fp, int space)
{
	nanxcf_write_int_prop(fp, XCF_PROP_COMPOSITE_SPACE, space);
}

void nanxcf_write_prop_compression_rle(FILE * fp)
{
	nanxcf_write_uint(fp, XCF_PROP_COMPRESSION);
	nanxcf_write_uint(fp, 1u);
	nanxcf_write_byte(fp, (unsigned char) 1);
}

void nanxcf_write_prop_end(FILE * fp)
{
	fwrite(&ZERO, sizeof(ZERO), 1u, fp);
	fwrite(&ZERO, sizeof(ZERO), 1u, fp);
}

void nanxcf_write_prop_float_opacity(FILE * fp, float opacity)
{
	assert(opacity >= 0 && opacity <= 1); /* xcf-write.ngg:320 */
	nanxcf_write_uint(fp, XCF_PROP_FLOAT_OPACITY);
	nanxcf_write_uint(fp, 4u);
	nanxcf_write_float(fp, opacity);
}

void nanxcf_write_prop_layer_mode(FILE * fp, unsigned int mode)
{
	nanxcf_write_uint_prop(fp, XCF_PROP_MODE, mode);
}

void nanxcf_write_prop_offsets_0_0(FILE * fp)
{
	nanxcf_write_uint(fp, XCF_PROP_OFFSETS);
	nanxcf_write_uint(fp, 8u);
	nanxcf_write_int(fp, 0);
	nanxcf_write_int(fp, 0);
}

void nanxcf_write_prop_visible(FILE * fp, _Bool visible)
{
	nanxcf_write_uint_prop(fp, XCF_PROP_VISIBLE, (unsigned int) visible);
}

void nanxcf_write_relptr(FILE * fp, long long offset)
{
	long curpos = ftell(fp);
	long long ptr = GINT64_TO_BE(((long long) curpos) + offset);
	nanxcf_write_absptrbe(fp, ptr);
}

void nanxcf_write_string(FILE * fp, const char * s)
{
	size_t slen = strlen(s);
	unsigned int finalsiz = htonl((unsigned int) (slen + 1));
	fwrite(&finalsiz, sizeof(finalsiz), 1u, fp);
	fwrite(s, slen + (1u), 1u, fp);
}

void nanxcf_write_uint(FILE * fp, unsigned int n)
{
	unsigned int finaln = htonl(n);
	fwrite(&finaln, sizeof(finaln), 1u, fp);
}

void nanxcf_write_uint_prop(FILE * fp, unsigned int propid, unsigned int value)
{
	nanxcf_write_uint(fp, propid);
	nanxcf_write_uint(fp, 4u);
	nanxcf_write_uint(fp, value);
}

void nanxcf_write_int_prop(FILE * fp, unsigned int propid, int value)
{
	nanxcf_write_uint(fp, propid);
	nanxcf_write_uint(fp, 4u);
	nanxcf_write_int(fp, value);
}

unsigned char get_byte_from_float(float f, int i)
{
	unsigned char arr[4];
	memcpy(arr, (const char *) (&f), 4u);
	return arr[i];
}

PointerReplacement pointer_replacement_default()
{
	PointerReplacement s;
	s.ptrloc = 0;
	s.ptrval = 0;
	return s;
}
